<?php
namespace App;
class Encoder {
    private string $source;
    private string $output;
    private string $buildId;
    private string $masterKey;
    public array $log = [];

    public function __construct(string $source, string $output) {
        $this->source = rtrim($source,'/');
        $this->output = rtrim($output,'/');
        $this->buildId = bin2hex(random_bytes(8));
        $this->masterKey = random_bytes(SODIUM_CRYPTO_AEAD_XCHACHA20POLY1305_IETF_KEYBYTES);
    }
    public function getBuildId(): string { return $this->buildId; }

    public function encodeAll(array $options): array {
        $files = $this->iteratePhp($this->source);
        $summary = ['files'=>0,'ok'=>0,'fail'=>0];
        foreach ($files as $idx=>$file) {
            $ok = $this->encodeFile($idx, $file, $options);
            $summary['files']++; $summary[$ok?'ok':'fail']++;
        }
        return $summary;
    }

    private function iteratePhp(string $dir): array {
        $rii = new \RecursiveIteratorIterator(new \RecursiveDirectoryIterator($dir,\FilesystemIterator::SKIP_DOTS));
        $out=[]; foreach ($rii as $f) { if (strtolower($f->getExtension())==='php') $out[]=$f->getPathname(); }
        sort($out); return $out;
    }

    private function minify(string $code): string {
        $tokens = token_get_all($code); $out='';
        foreach ($tokens as $tok){
            if (is_array($tok)){
                [$id,$text] = $tok;
                if ($id===T_COMMENT || $id===T_DOC_COMMENT) continue;
                if ($id===T_WHITESPACE){ $out.=' '; continue; }
                $out.=$text;
            } else { $out.=$tok; }
        }
        // Tiny rename for common vars
        $out = preg_replace('/\$(i|j|k)\b/','$_v'.bin2hex(random_bytes(2)),$out);
        return $out;
    }

    private function relOut(string $in): string {
        $rel = ltrim(str_replace('\\','/', substr($in, strlen($this->source))), '/');
        $target = $this->output.'/'.$rel;
        @mkdir(dirname($target), 0775, true);
        return $target;
    }

    private function obfuscateKey(string $key): array {
        // split into 4 chunks and XOR with random masks; rotate bits by offsets
        $chunks = str_split($key, intdiv(strlen($key),4) ?: 8);
        $masks = [];
        $rot = [];
        $encParts = [];
        foreach ($chunks as $ch) {
            $mask = random_bytes(strlen($ch));
            $masks[] = base64_encode($mask);
            $r = random_int(1,7);
            $rot[] = $r;
            $x = $ch ^ $mask;
            // bit rotate left by r per byte
            $x2 = '';
            for ($i=0;$i<strlen($x);$i++){
                $b = ord($x[$i]);
                $x2 .= chr((($b << $r) & 0xFF) | ($b >> (8-$r)));
            }
            $encParts[] = base64_encode($x2);
        }
        return ['parts'=>$encParts,'masks'=>$masks,'rot'=>$rot];
    }

    public function encodeFile(int $idx, string $file, array $opt): bool {
        $raw = file_get_contents($file);
        if ($raw===false){ $this->log[]="✖ read fail: $file"; return false; }
        $code = !empty($opt['obf']) ? $this->minify($raw) : $raw;

        $fileKey = sodium_crypto_kdf_derive_from_key(
            SODIUM_CRYPTO_AEAD_XCHACHA20POLY1305_IETF_KEYBYTES,
            $idx,
            substr(hash('sha256', $this->buildId, true), 0, SODIUM_CRYPTO_KDF_KEYBYTES),
            $this->masterKey
        );
        $nonce = random_bytes(SODIUM_CRYPTO_AEAD_XCHACHA20POLY1305_IETF_NPUBBYTES);
        $meta = [
            'lic_mode'=>$opt['lic_mode']??'local','lic_url'=>$opt['lic_url']??'','lic_app'=>$opt['lic_app']??'','lic_pub'=>$opt['lic_pub']??'','lic_cache'=>(int)($opt['lic_cache']??3600),'build'=>$this->buildId, 'path'=>str_replace('\\','/',$file), 'ts'=>time(),
            'expires'=>$opt['expires']??0, 'domains'=>$opt['domains']??[], 'online'=>$opt['online']??0,
            'wp'=>$opt['wp']??0, 'algo'=>'xchacha20poly1305_ietf'
        ];
        $aad = json_encode($meta, JSON_UNESCAPED_SLASHES);
        $ct = sodium_crypto_aead_xchacha20poly1305_ietf_encrypt($code, $aad, $nonce, $fileKey);

        // Signature (Ed25519) over AAD||CT for tamper detection
        $signKP = sodium_crypto_sign_keypair();
        $pk = sodium_crypto_sign_publickey($signKP);
        $sk = sodium_crypto_sign_secretkey($signKP);
        $sig = sodium_crypto_sign_detached($aad.$ct, $sk);

        // Obfuscate per-file key
        $keyObf = $this->obfuscateKey($fileKey);

        $stub = file_get_contents(__DIR__.('/../stubs/'.(($opt['lic_mode']??'local')==='server'?'stub_license.php':'stub_pro.php')));
        if ($stub===false){ $this->log[]="✖ missing stub"; return false; }

        $payload = str_replace(
            ['__KEY_PARTS__','__KEY_MASKS__','__KEY_ROT__','__NONCE__','__AAD__','__CT__','__DOMAINS__','__EXPIRES__','__ONLINE__','__BUILD__','__IS_WP__','__ERRMSG__','__PUBKEY__','__SIGN__'],
            [
                json_encode($keyObf['parts']),
                json_encode($keyObf['masks']),
                json_encode($keyObf['rot']),
                base64_encode($nonce),
                base64_encode($aad),
                base64_encode($ct),
                base64_encode(json_encode($opt['domains']??[], JSON_UNESCAPED_SLASHES)),
                (string)($opt['expires']??0),
                (string)($opt['online']??0),
                $this->buildId,
                (string)($opt['wp']??0),
                base64_encode($opt['errmsg']??'Access denied.'),
                base64_encode($pk),
                base64_encode($sig)
            ],
            $stub
        );

        $out = $this->relOut($file);
        $ok = (bool)file_put_contents($out, $payload);
        $this->log[] = ($ok?'✔':'✖') . ' ' . $file;
        return $ok;
    }
}
